//----------------------------------------------------------------------------
// Copyright 2001-2021 Joe Lowe
//
// Permission is granted to any person obtaining a copy of this Software,
// to deal in the Software without restriction, including the rights to use,
// copy, modify, merge, publish, distribute, sublicense, and sell copies of
// the Software.
//
// The above copyright and permission notice must be left intact in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS WITHOUT WARRANTY.
//----------------------------------------------------------------------------
// file name:   traces.h
// created:     2001.05.12
//----------------------------------------------------------------------------
#ifndef TRACES_H
#define TRACES_H
#if ! defined KERNEL || ! defined __linux__
#include <stdarg.h>
#endif
#include "ptpublic.h"
#ifdef __cplusplus
 #define _TRWCHAR wchar_t
#elif defined(__WCHAR_TYPE__)
 #define _TRWCHAR __WCHAR_TYPE__
#else
 #include <wchar.h>
 #define _TRWCHAR wchar_t
#endif
#define _TRCDECL PT_CCALL
#define _TRPRIV PT_PRIVATE
#define _TRINLINE PT_FORCEINLINE
#define _TRINLINEV PT_INLINE2

   // This header is compatible with C and C++ compilers.
#ifdef __cplusplus
extern "C" {
#endif

   // Opaque channel handle.
typedef struct _TRCHANNEL* TRCHANNEL;

   // Predefined handle for default channel.
#define TRDEF ((TRCHANNEL)((PT_SIZE_T)(0)-1))
   // All functions will perform no action if passed a null handle.
#define TRNULL ((TRCHANNEL)(0))

   // Open and close handle to channel. Application code can call open and
   // close at any time. Driver code must call open and close from known
   // system thread context at APC or lower IRQL. It is not necessary to
   // check the return value from the open or dup functions.
_TRPRIV TRCHANNEL _TRCDECL tropen(const char* channelname);
_TRPRIV TRCHANNEL _TRCDECL trwopen(const _TRWCHAR* channelname);
_TRPRIV TRCHANNEL _TRCDECL trdup(TRCHANNEL);
_TRPRIV void _TRCDECL trclose(TRCHANNEL);
   // Set channel to use when TRDEF is specified.
_TRPRIV void _TRCDECL trdefault(TRCHANNEL);
   // Clear data in trace channel.
_TRPRIV void _TRCDECL trreset(TRCHANNEL);
   // Attach and detach calls establish and terminate a connection with the
   // trace driver. For application code the trace import library performs an
   // automatic attach at process or dll initialization. A matching detach is
   // performed automatically at process exit or dll unload. Driver code must
   // manually call attach and detach functions from a known system thread
   // context at APC or lower IRQL. It is not necessary to check the return
   // result from the attach function. The attach function call can be issued
   // periodically to allow for dynamic attach to the trace driver if it was
   // not available at initialization. The client code is responsible to
   // serialize the detach call with all other calls.
_TRPRIV int/*bool attached*/ _TRCDECL trattach(void);
_TRPRIV void _TRCDECL trdetach(void);

   // Don't use this function. You won't like it.
_TRPRIV void _TRCDECL trcrash(void);

   // Functions for outputting formatted data to trace channel. All calls are
   // fully reentrant for application and driver code. Driver code must call
   // output functions from DISPATCH or lower IRQL.
_TRPRIV void _TRCDECL trprint(TRCHANNEL,const char* text);
_TRPRIV void _TRCDECL trwprint(TRCHANNEL,const _TRWCHAR* text);
_TRPRIV void _TRCDECL trvprintf(TRCHANNEL,const char* format,va_list args);
_TRPRIV void _TRCDECL trprintf(TRCHANNEL,const char* format,...);
_TRPRIV void _TRCDECL trvwprintf(TRCHANNEL,const _TRWCHAR* format,va_list args);
_TRPRIV void _TRCDECL trwprintf(TRCHANNEL,const _TRWCHAR* format,...);

   // Functions to read channel data for use by trace display utilities.
   // Driver code must call read functions from known system thread context
   // at APC or lower IRQL.
_TRPRIV void _TRCDECL trlist(char* buffer,PT_SIZE_T maxBufferChars);
_TRPRIV void _TRCDECL trwlist(_TRWCHAR* buffer,PT_SIZE_T maxBufferChars);
_TRPRIV void _TRCDECL trinfo(TRCHANNEL,PT_SIZE_T* startLine,PT_SIZE_T* endLine,PT_SIZE_T* endLineLen);
_TRPRIV void _TRCDECL trread(TRCHANNEL,PT_SIZE_T line,char* buffer,PT_SIZE_T maxBufferChars);
_TRPRIV void _TRCDECL trwread(TRCHANNEL,PT_SIZE_T line,_TRWCHAR* buffer,PT_SIZE_T maxBufferChars);

#ifdef __cplusplus
}

   // Argument overloaded functions so C++ code doesn't need to use wide char
   // specific function names.
_TRINLINE TRCHANNEL tropen(const _TRWCHAR* channelName)
   { return trwopen(channelName); }
_TRINLINE void trprint(TRCHANNEL channel,const _TRWCHAR* text)
   { trwprint(channel,text); }
_TRINLINE void trvprintf(TRCHANNEL channel,const _TRWCHAR* format,va_list args)
   { trvwprintf(channel,format,args); }
_TRINLINEV void trprintf(TRCHANNEL channel,const _TRWCHAR* format,...)
   { va_list args; va_start(args,format); trvwprintf(channel,format,args); va_end(args); }
_TRINLINE void trlist(_TRWCHAR* buffer,PT_SIZE_T maxBufferChars)
   { trwlist(buffer,maxBufferChars); }
_TRINLINE void trread(TRCHANNEL channel,PT_SIZE_T line,_TRWCHAR* buffer,PT_SIZE_T maxBufferChars)
   { trwread(channel,line,buffer,maxBufferChars); }

    // Argument overloaded functions so C++ code doesn't need to specify
    // default channel.
_TRINLINE void trprint(const char* text)
   { trprint(TRDEF,text); }
_TRINLINE void trvprintf(const char* format,va_list args)
   { trvprintf(TRDEF,format,args); }
_TRINLINEV void trprintf(const char* format,...)
   { va_list args; va_start(args,format); trvprintf(TRDEF,format,args); va_end(args); }
_TRINLINE void trprint(const _TRWCHAR* text)
   { trwprint(TRDEF,text); }
_TRINLINE void trvprintf(const _TRWCHAR* format,va_list args)
   { trvwprintf(TRDEF,format,args); }
_TRINLINEV void trprintf(const _TRWCHAR* format,...)
   { va_list args; va_start(args,format); trvwprintf(TRDEF,format,args); va_end(args); }
_TRINLINE void trread(PT_SIZE_T line,char* buffer,PT_SIZE_T maxBufferChars)
   { trread(TRDEF,line,buffer,maxBufferChars); }
_TRINLINE void trread(PT_SIZE_T line,_TRWCHAR* buffer,PT_SIZE_T maxBufferChars)
   { trwread(TRDEF,line,buffer,maxBufferChars); }

   // Bogus channel class to allow channel->print(...) style function
   // invocation in C++ code.
struct _TRCHANNEL
{
private:
   int unused;
      // This isn't a real class so inhibit copies and construction.
   _TRCHANNEL(void) { ; }
   _TRCHANNEL(TRCHANNEL&) { ; }
   _TRCHANNEL& operator=(_TRCHANNEL&) { return *this; }
public:
   inline void close(void);
   inline TRCHANNEL dup(void);
   inline void print(const char* text);
   inline void vprintf(const char* format,va_list args);
   inline void printf(const char* format,va_list args);
   inline void printf(const char* format,...);
   inline void wprint(const _TRWCHAR* text);
   inline void print(const _TRWCHAR* text);
   inline void vwprintf(const _TRWCHAR* format,va_list args);
   inline void vprintf(const _TRWCHAR* format,va_list args);
   inline void printf(const _TRWCHAR* format,va_list args);
   inline void wprintf(const _TRWCHAR* format,...);
   inline void printf(const _TRWCHAR* format,...);
   inline void info(PT_SIZE_T* startLine,PT_SIZE_T* endLine,PT_SIZE_T* endLineLen);
   inline void read(PT_SIZE_T line,char* buffer,PT_SIZE_T maxBufferChars);
   inline void read(PT_SIZE_T line,_TRWCHAR* buffer,PT_SIZE_T maxBufferChars);
};

_TRPRIV _TRINLINE void _TRCHANNEL::close(void)
   { trclose(this); }
_TRPRIV _TRINLINE TRCHANNEL _TRCHANNEL::dup(void)
   { return trdup(this); }
_TRPRIV _TRINLINE void _TRCHANNEL::print(const char* text)
   { trprint(this,text); }
_TRPRIV _TRINLINE void _TRCHANNEL::vprintf(const char* format,va_list args)
   { trvprintf(this,format,args); }
_TRPRIV _TRINLINE void _TRCHANNEL::printf(const char* format,va_list args)
   { trvprintf(this,format,args); }
_TRPRIV inline void _TRCHANNEL::printf(const char* format,...)
   { va_list args; va_start(args,format); trvprintf(this,format,args); va_end(args); }
_TRPRIV _TRINLINE void _TRCHANNEL::wprint(const _TRWCHAR* text)
   { trwprint(this,text); }
_TRPRIV _TRINLINE void _TRCHANNEL::print(const _TRWCHAR* text)
   { trwprint(this,text); }
_TRPRIV _TRINLINE void _TRCHANNEL::vwprintf(const _TRWCHAR* format,va_list args)
   { trvwprintf(this,format,args); }
_TRPRIV _TRINLINE void _TRCHANNEL::vprintf(const _TRWCHAR* format,va_list args)
   { trvwprintf(this,format,args); }
_TRPRIV _TRINLINE void _TRCHANNEL::printf(const _TRWCHAR* format,va_list args)
   { trvwprintf(this,format,args); }
_TRPRIV inline void _TRCHANNEL::wprintf(const _TRWCHAR* format,...)
   { va_list args; va_start(args,format); trvwprintf(this,format,args); va_end(args); }
_TRPRIV inline void _TRCHANNEL::printf(const _TRWCHAR* format,...)
   { va_list args; va_start(args,format); trvwprintf(this,format,args); va_end(args); }
_TRPRIV _TRINLINE void _TRCHANNEL::info(PT_SIZE_T* startLine,PT_SIZE_T* endLine,PT_SIZE_T* endLineLen)
   { trinfo(this,startLine,endLine,endLineLen); }
_TRPRIV _TRINLINE void _TRCHANNEL::read(PT_SIZE_T line,char* buffer,PT_SIZE_T maxBufferChars)
   { trread(this,line,buffer,maxBufferChars); }
_TRPRIV _TRINLINE void _TRCHANNEL::read(PT_SIZE_T line,_TRWCHAR* buffer,PT_SIZE_T maxBufferChars)
   { trwread(this,line,buffer,maxBufferChars); }

#endif

#endif
